package com.keymobile.governworkorder.core.service.impl;

import com.keymobile.governworkorder.common.bo.WorkOrderDataVO;
import com.keymobile.governworkorder.common.bo.WorkOrderInfoVO;
import com.keymobile.governworkorder.common.bo.WorkOrderProcessVO;
import com.keymobile.governworkorder.common.bo.WorkOrderStageVO;
import com.keymobile.governworkorder.common.type.*;
import com.keymobile.governworkorder.core.dto.WorkOrdeQueryDto;
import com.keymobile.governworkorder.core.dto.WorkOrderConfirmDataDto;
import com.keymobile.governworkorder.core.messaging.WorkOrderMessageService;
import com.keymobile.governworkorder.core.persistence.model.WorkOrderNotice;
import com.keymobile.governworkorder.core.persistence.model.WorkOrderProcess;
import com.keymobile.governworkorder.core.persistence.model.WorkOrderData;
import com.keymobile.governworkorder.core.persistence.model.WorkOrderInfo;
import com.keymobile.governworkorder.core.remote.AuthService;
import com.keymobile.governworkorder.core.remote.user.User;
import com.keymobile.governworkorder.core.remote.user.UserGroup;
import com.keymobile.governworkorder.core.service.FileService;
import com.keymobile.governworkorder.core.service.NoticeService;
import com.keymobile.governworkorder.core.service.WorkOrderService;
import com.keymobile.governworkorder.core.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.support.PageableExecutionUtils;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author xiesh
 * @version 1.0.0
 * @date 2025/8/13
 * @desc
 */
@Service
@Slf4j
public class WorkOrderServiceImpl implements WorkOrderService {

    @Autowired
    private MongoTemplate mongoTemplate;
    @Autowired
    private WorkOrderMessageService messageService;
    @Autowired
    private FileService fileService;
    @Autowired
    private AuthService authService;
    @Autowired
    private NoticeService noticeService;
    @Value("${process.createGroup:治理工单创建组}")
    private String createGroup;
    @Value("${process.approvalGroup:治理工单审批组}")
    private String approvalGroup;
    @Value("${process.superAdmin:root}")
    private String superAdmin;


    public Page<WorkOrderInfoVO> page(WorkOrdeQueryDto param) {
        Pageable pageable = PageRequest.of(param.getPage() - 1, param.getSize(),
                param.getDirection(), param.getSort());
        Query query = new Query();
        if (StringUtils.isNotBlank(param.getType())) {
            query.addCriteria(Criteria.where("type").is(param.getType()));
        }
        if (StringUtils.isNotBlank(param.getLevel())) {
            query.addCriteria(Criteria.where("level").is(param.getLevel()));
        }
        if (StringUtils.isNotBlank(param.getStatus())) {
            query.addCriteria(Criteria.where("status").is(param.getStatus()));
        }
        if (StringUtils.isNotBlank(param.getDataType())) {
            query.addCriteria(Criteria.where("dataType").is(param.getDataType()));
        }
        if (StringUtils.isNotBlank(param.getStage())) {
            query.addCriteria(Criteria.where("stage").is(param.getStage()));
        }
        if (StringUtils.isNotBlank(param.getKeyword())) {
            query.addCriteria(new Criteria().orOperator(
                    Criteria.where("desc").regex(param.getKeyword()),
                    Criteria.where("name").regex(param.getKeyword()),
                    Criteria.where("userDname").regex(param.getKeyword()),
                    Criteria.where("handleUserDname").regex(param.getKeyword()),
                    Criteria.where("title").regex(param.getKeyword())));
        }

        // 添加时间范围查询
        if (StringUtils.isNotBlank(param.getStartTime()) || StringUtils.isNotBlank(param.getEndTime())) {
            Criteria timeCriteria = Criteria.where("createTimeMill");
            if (StringUtils.isNotBlank(param.getStartTime())) {
                try {
                    LocalDate startLocalDate = LocalDate.parse(param.getStartTime());
                    long startTimeMillis = startLocalDate.atStartOfDay(ZoneId.of("+8"))
                            .toInstant()
                            .toEpochMilli();
                    timeCriteria.gte(startTimeMillis);
                } catch (Exception e) {
                    log.warn("开始时间格式错误: {}", param.getStartTime());
                }
            }

            if (StringUtils.isNotBlank(param.getEndTime())) {
                try {
                    LocalDate endLocalDate = LocalDate.parse(param.getEndTime());
                    endLocalDate = endLocalDate.plusDays(1);
                    long endTimeMillis = endLocalDate.atStartOfDay(ZoneId.of("+8"))
                            .toInstant()
                            .toEpochMilli();
                    timeCriteria.lte(endTimeMillis);
                } catch (Exception e) {
                    log.warn("结束时间格式错误: {}", param.getEndTime());
                }
            }
            query.addCriteria(timeCriteria);

        }
        String userName = LogManager.getUserName();
        //待办
        if (StringUtils.equals(param.getDoType(), "1")) {
            query.addCriteria(Criteria.where("nextUserName").regex("(^|,)"+userName+"(,|$)"));
            //已办
        } else if (StringUtils.equals(param.getDoType(), "2")) {
            Query processQuery = new Query();
            processQuery.addCriteria(Criteria.where("userName").is(userName));
            processQuery.fields().include("workOrderId");
            List<String> processDocs = mongoTemplate.findDistinct(processQuery, "workOrderId", WorkOrderProcess.class, String.class);
            if (CollectionUtils.isEmpty(processDocs)) {
                return PageableExecutionUtils.getPage(Collections.emptyList(), pageable, () -> 0);
            }
            query.addCriteria(Criteria.where("_id").in(processDocs));
        } else {
            //如果非待办和已办，则根据账号权限查询关联数据。 自理工单创建组成员可以查看所有数据，其他则只能查看自己相关数据（待办、已办）
            boolean createUserFlag = checkCreatePermission(userName);
            if (!createUserFlag) {
                Query processQuery = new Query();
                processQuery.addCriteria(Criteria.where("userName").is(userName));
                processQuery.fields().include("workOrderId");
                List<String> processDocs = mongoTemplate.findDistinct(processQuery, "workOrderId", WorkOrderProcess.class, String.class);
                if (CollectionUtils.isEmpty(processDocs)) {
                    query.addCriteria(Criteria.where("nextUserName").regex("(^|,)" + userName + "(,|$)"));
                } else {
                    query.addCriteria(new Criteria().orOperator(
                            Criteria.where("nextUserName").regex("(^|,)" + userName + "(,|$)"),
                            Criteria.where("_id").in(processDocs)));
                }
            }
        }
        long count = mongoTemplate.count(query, WorkOrderInfo.class);
        query.with(pageable);
        List<WorkOrderInfo> list = mongoTemplate.find(query, WorkOrderInfo.class);
        List<WorkOrderInfoVO> voList = list.stream().map(e -> BeanUtil.map(e, WorkOrderInfoVO.class))
                .collect(Collectors.toList());
        return PageableExecutionUtils.getPage(voList, pageable, () -> count);

    }


    @Override
    public WorkOrderInfoVO add(WorkOrderInfoVO workOrderVo) {
        WorkOrderInfo workOrder = BeanUtil.map(workOrderVo, WorkOrderInfo.class);
        List<WorkOrderDataVO> dataVOList = workOrderVo.getData();
        if (!CollectionUtils.isEmpty(dataVOList)) {
            List<WorkOrderData> dataList = dataVOList.stream().map(e -> BeanUtil.map(e, WorkOrderData.class)).collect(Collectors.toList());
            workOrder.setData(dataList);
        }
        validateAndFillWorkOrder(workOrder);
        workOrder = mongoTemplate.save(workOrder);
        workOrder.setCreateTime(DateUtil.getDateTime());
        workOrder.setCreateTimeMill(System.currentTimeMillis());
        if (!CollectionUtils.isEmpty(workOrder.getData())) {
            for (WorkOrderData data : workOrder.getData()) {
                fillWorkOrderData(workOrder, data);
                mongoTemplate.save(data);
            }
        }
        //启动流程
        startProcess(workOrder.getId());

        return getById(workOrder.getId());
    }

    @Override
    public List<WorkOrderDataVO> saveData(String workOrderId, List<WorkOrderDataVO> workOrderDataVOList) {

        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        //非发起人和处理人无法操作
        if (!contains(workOrder.getNextUserName(), LogManager.getUserName())
                || !StringUtils.equals(LogManager.getUserName(), workOrder.getHandleUserName())) {
            throw new RuntimeException("您无权限编辑关联数据");
        }

        //删除旧数据
        Query query = Query.query(Criteria.where("workOrderId").is(workOrderId));
        mongoTemplate.findAllAndRemove(query, WorkOrderData.class);

        List<WorkOrderDataVO> result = new ArrayList<>();
        for (WorkOrderDataVO workOrderDataVO : workOrderDataVOList) {
            WorkOrderData workOrderData = BeanUtil.map(workOrderDataVO, WorkOrderData.class);
            fillWorkOrderData(workOrder, workOrderData);
            workOrderData = mongoTemplate.save(workOrderData);
            result.add(BeanUtil.map(workOrderData, WorkOrderDataVO.class));
        }
        return result;

    }

    @Override
    public void confirmData(String workOrderId, List<WorkOrderConfirmDataDto> confirmDataList) {
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        if (!StringUtils.equals(OrderProcessStageTypeEnum.PROCESS_APPROVER.getName(), workOrder.getStage())
                || !contains(workOrder.getNextUserName(), LogManager.getUserName())) {
            throw new RuntimeException("非审批人无法确认工单");
        }
        for (WorkOrderConfirmDataDto item : confirmDataList) {
            WorkOrderData workOrderData = mongoTemplate.findById(item.getId(), WorkOrderData.class);
            if (workOrderData == null) {
                log.error("未找到对应关联数据：{}", ObjectUtil.toJson(item));
                continue;
            }
            workOrderData.setConfirmStatus(item.getConfirmStatus());
            mongoTemplate.save(workOrderData);
        }

    }

    public void delete(String workOrderId, boolean force) {
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        if (!force) {
            //非发起人和处理人无法操作
            if (!StringUtils.equals(LogManager.getUserName(), workOrder.getCreateUserName())) {
                throw new RuntimeException("非发起人无法删除");
            }
            //非发起人和处理人无法操作
            if (!StringUtils.equals(workOrder.getStatus(), OrderStatusTypeEnum.STATUS_TO_BE_START.getName())) {
                throw new RuntimeException("非待发起状态的工单无法删除");
            }
        }

        //发送消息
        WorkOrderInfoVO messageVO = convertMessage(workOrder);
        messageService.sendMsg(OrderMessageTypeEnum.MSG_DELETE.getType(), messageVO);

        //删除关联数据
        String fileId = workOrder.getFileId();
        fileService.del(fileId);
        Query query = Query.query(Criteria.where("workOrderId").is(workOrderId));
        mongoTemplate.remove(query, WorkOrderProcess.class);
        mongoTemplate.remove(query, WorkOrderData.class);
        mongoTemplate.remove(Query.query(Criteria.where("_id").is(workOrderId)), WorkOrderInfo.class);

    }

    @Override
    public List<WorkOrderDataVO> getData(String workOrderId, String namePath, String keyword) {
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        Query query = Query.query(Criteria.where("workOrderId").is(workOrderId));
        if (StringUtils.isNotBlank(namePath)) {
            query.addCriteria(Criteria.where("namePath").regex(namePath));
        }
        if (StringUtils.isNotBlank(keyword)) {
            query.addCriteria(new Criteria().orOperator(
                    Criteria.where("name").regex(keyword),
                    Criteria.where("cnName").regex(keyword),
                    Criteria.where("code").regex(keyword),
                    Criteria.where("desc").regex(keyword)));
        }
        List<WorkOrderData> list = mongoTemplate.find(query, WorkOrderData.class);
        return list.stream().map(e -> BeanUtil.map(e, WorkOrderDataVO.class)).collect(Collectors.toList());
    }

    @Override
    public List<WorkOrderProcessVO> getProcesses(String workOrderId) {
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        Query query = Query.query(Criteria.where("workOrderId").is(workOrderId));
        List<WorkOrderProcess> list = mongoTemplate.find(query, WorkOrderProcess.class);
        return list.stream().map(e -> BeanUtil.map(e, WorkOrderProcessVO.class)).collect(Collectors.toList());
    }

    @Override
    public WorkOrderInfoVO getById(String workOrderId) {
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        Query query = Query.query(Criteria.where("workOrderId").is(workOrderId));
        List<WorkOrderData> dataList = mongoTemplate.find(query, WorkOrderData.class);
        List<WorkOrderProcess> processList = mongoTemplate.find(query, WorkOrderProcess.class);
        WorkOrderInfoVO workOrderVO = BeanUtil.map(workOrder, WorkOrderInfoVO.class);

        if (!CollectionUtils.isEmpty(dataList)) {
            List<WorkOrderDataVO> dataVOList = dataList.stream().map(e -> BeanUtil.map(e, WorkOrderDataVO.class)).collect(Collectors.toList());
            workOrderVO.setData(dataVOList);
        }
        if (!CollectionUtils.isEmpty(processList)) {
            List<WorkOrderProcessVO> dataVOList = processList.stream().map(e -> BeanUtil.map(e, WorkOrderProcessVO.class)).collect(Collectors.toList());
            workOrderVO.setProcess(dataVOList);
        }
        return workOrderVO;
    }

    @Override
    public boolean checkCreatePermission(String userName) {
        String userNameTemp;
        if (StringUtils.isBlank(userName)) {
            userNameTemp = LogManager.getUserName();
        } else {
            userNameTemp = userName;
        }
        if (checkSuperAdmin(userNameTemp)) {
            return true;
        }
        List<User> users = getUserNameByGroup(createGroup);
        User user = users.stream().filter(e -> StringUtils.equals(userNameTemp, e.getName())).findFirst().orElse(null);
        return !(user == null);
    }

    @Override
    public void startProcess(String workOrderId) {
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        if (!StringUtils.equals(LogManager.getUserName(), workOrder.getCreateUserName())) {
            throw new RuntimeException("非本人无法发起工单流程");
        }
        if (!StringUtils.equals(OrderStatusTypeEnum.STATUS_TO_BE_START.getName(), workOrder.getStatus())) {
            throw new RuntimeException("非待发起的工单无法启动流程");
        }
        WorkOrderProcess process = new WorkOrderProcess();
        process.setId(IdUtil.generateId());
        process.setUserName(LogManager.getUserName());
        process.setUserDname(LogManager.getDUserName());
        process.setStage(OrderProcessStageTypeEnum.PROCESS_CREATE.getName());
        process.setContent("发起流程");
        process.setType(OrderApprovalTypeEnum.APPROVAL_PASS.getName());
        process.setNextUserName(workOrder.getHandleUserName());
        process.setNextUserDname(workOrder.getHandleUserDname());
        process.setCreateTime(DateUtil.getDateTime());
        process.setWorkOrderId(workOrder.getId());
        mongoTemplate.save(process);

        workOrder.setStatus(OrderStatusTypeEnum.STATUS_PROCESSING.getName());
        workOrder.setStage(OrderProcessStageTypeEnum.PROCESS_HANDLER.getName());
        workOrder.setUpdateTime(DateUtil.getDateTime());
        workOrder.setNextUserName(workOrder.getHandleUserName());
        workOrder.setNextUserDname(workOrder.getHandleUserDname());
        mongoTemplate.save(workOrder);

        //发送消息
        WorkOrderInfoVO messageVO = convertMessage(workOrder);
        messageService.sendMsg(OrderMessageTypeEnum.MSG_CREATE.getType(), messageVO);

    }

    @Override
    public List<User> findUser(String[] types, Long[] dataRoleIds, String match, Long[] userGroupIds) {
        return authService.find(types, dataRoleIds, match, userGroupIds);
    }

    private boolean checkSuperAdmin(String userName) {
        return StringUtils.equals(superAdmin, userName);
    }

    public void approvalProcess(String workOrderId, WorkOrderProcessVO processVO) {
        log.debug("workOrderId:{},process:{}", workOrderId, ObjectUtil.toJson(processVO));
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");

        if (!StringUtils.equals(OrderStatusTypeEnum.STATUS_PROCESSING.getName(), workOrder.getStatus())) {
            throw new RuntimeException(workOrder.getStatus() + "，无法审批");
        }

        if (!contains(workOrder.getNextUserName(), LogManager.getUserName())) {
            throw new RuntimeException("无权限发起审批");
        }

        //处理人无法驳回
        if (StringUtils.equals(OrderProcessStageTypeEnum.PROCESS_HANDLER.getName(), workOrder.getStage())
                && StringUtils.equals(OrderApprovalTypeEnum.APPROVAL_REJECT.getName(), processVO.getType())) {
            throw new RuntimeException("处理人无法驳回");
        }

        WorkOrderProcess process = BeanUtil.map(processVO, WorkOrderProcess.class);
        process.setId(IdUtil.generateId());
        process.setUserName(LogManager.getUserName());
        process.setUserDname(LogManager.getDUserName());
        process.setStage(workOrder.getStage());
        process.setCreateTime(DateUtil.getDateTime());
        process.setWorkOrderId(workOrder.getId());

        //驳回
        if (StringUtils.equals(OrderApprovalTypeEnum.APPROVAL_REJECT.getName(), processVO.getType())) {
            //资产完善工单审批人驳回返回处理人处理，否则工单失效
            if (StringUtils.equals(OrderTypeEnum.DATA_UPDATE.getName(), workOrder.getType())) {
                process.setNextUserName(workOrder.getHandleUserName());
                process.setNextUserDname(workOrder.getHandleUserDname());
                workOrder.setNextUserName(workOrder.getHandleUserName());
                workOrder.setNextUserDname(workOrder.getHandleUserDname());
                workOrder.setStage(OrderProcessStageTypeEnum.PROCESS_HANDLER.getName());
                mongoTemplate.save(process);
                mongoTemplate.save(workOrder);
            } else {
                //驳回 工单完成，但是发失效消息
                process.setNextUserName(null);
                process.setNextUserDname(null);
                workOrder.setNextUserName(null);
                workOrder.setNextUserDname(null);
                workOrder.setStage(OrderProcessStageTypeEnum.PROCESS_DONE.getName());
                workOrder.setStatus(OrderStatusTypeEnum.STATUS_INVALIDATE.getName());
                workOrder.setEndTime(DateUtil.getDate());
                workOrder.setEndTimeMill(System.currentTimeMillis());
                mongoTemplate.save(process);
                mongoTemplate.save(workOrder);
                WorkOrderInfoVO messageVO = convertMessage(workOrder);
                messageService.sendMsg(OrderMessageTypeEnum.MSG_INVALIDATE.getType(), messageVO);
            }

            //通过
        } else {
            //审批人通过，工单结束，发送消息
            if (StringUtils.equals(OrderApprovalTypeEnum.APPROVAL_PASS.getName(), processVO.getType())
                    && StringUtils.equals(OrderProcessStageTypeEnum.PROCESS_APPROVER.getName(), workOrder.getStage())) {
                process.setNextUserName(null);
                process.setNextUserDname(null);
                workOrder.setNextUserName(null);
                workOrder.setNextUserDname(null);
                workOrder.setStage(OrderProcessStageTypeEnum.PROCESS_DONE.getName());
                workOrder.setStatus(OrderStatusTypeEnum.STATUS_DONE.getName());
                workOrder.setEndTime(DateUtil.getDate());
                workOrder.setEndTimeMill(System.currentTimeMillis());

                mongoTemplate.save(process);
                mongoTemplate.save(workOrder);

                WorkOrderInfoVO messageVO = convertMessage(workOrder);
                messageService.sendMsg(OrderMessageTypeEnum.MSG_FINISH.getType(), messageVO);

                //处理人通过，发送审批人审批
            } else if (StringUtils.equals(OrderApprovalTypeEnum.APPROVAL_PASS.getName(), processVO.getType())
                    && StringUtils.equals(OrderProcessStageTypeEnum.PROCESS_HANDLER.getName(), workOrder.getStage())) {
                List<User> users = getUserNameByGroup(approvalGroup);
                Assert.notEmpty(users, "未配置用户组：" + approvalGroup);
                List<String> userNameList = users.stream().map(User::getName).collect(Collectors.toUnmodifiableList());
                List<String> userDnameList = users.stream().map(User::getDname).collect(Collectors.toUnmodifiableList());
                String userNameStr = StringUtils.join(userNameList,",");
                String userDnameStr = StringUtils.join(userDnameList,",");

                process.setNextUserName(userNameStr);
                process.setNextUserDname(userDnameStr);
                workOrder.setNextUserName(userNameStr);
                workOrder.setNextUserDname(userDnameStr);
                workOrder.setStage(OrderProcessStageTypeEnum.PROCESS_APPROVER.getName());
                mongoTemplate.save(process);
                mongoTemplate.save(workOrder);
            }
        }


    }

    @Override
    public List<WorkOrderStageVO> stages(String workOrderId) {
        List<WorkOrderStageVO> result = new ArrayList<>();
        WorkOrderInfo workOrder = mongoTemplate.findById(workOrderId, WorkOrderInfo.class);
        Assert.notNull(workOrder, "找不到对应工单");
        String stage = workOrder.getStage();
        Query query = Query.query(Criteria.where("workOrderId").is(workOrderId));
        query.with(Sort.by(Sort.Direction.DESC, "createTime"));
        List<WorkOrderProcess> processList = mongoTemplate.find(query, WorkOrderProcess.class);
        boolean handle = true;
        for (OrderProcessStageTypeEnum item : OrderProcessStageTypeEnum.values()) {
            WorkOrderStageVO vo = new WorkOrderStageVO();
            String itemName = item.getName();
            vo.setName(itemName);
            if (StringUtils.equals(stage, itemName)) {
                vo.setUserName(workOrder.getNextUserName());
                vo.setUserDname(workOrder.getNextUserDname());
                vo.setCurrent(true);
                vo.setHandle(false);
                handle = false;
            } else if (handle) {
                vo.setCurrent(false);
                vo.setHandle(handle);
                if (!CollectionUtils.isEmpty(processList)) {
                    WorkOrderProcess process = processList.stream().filter(e -> StringUtils.equals(e.getStage(), itemName))
                            .findFirst().orElse(null);
                    if (process != null) {
                        vo.setUserName(process.getUserName());
                        vo.setUserDname(process.getUserDname());
                        vo.setCreateTime(process.getCreateTime());
                    }
                }

            } else {
                vo.setCurrent(false);
                vo.setHandle(handle);
            }
            result.add(vo);
        }
        return result;

    }

    @Scheduled(cron = "0 0 6 * * ?")
    public void timeoutNotice() {
        log.info("超时通知同步开始");
        long now = LocalDate.now(ZoneId.of("+8"))
                .atStartOfDay()
                .toInstant(ZoneOffset.of("+8")).toEpochMilli();

        Query query = Query.query(Criteria.where("status").is(OrderStatusTypeEnum.STATUS_PROCESSING.getName())
                .and("expectedEndTime").lt(String.valueOf(now)));
        List<WorkOrderInfo> workOrderList = mongoTemplate.find(query, WorkOrderInfo.class);
        for (WorkOrderInfo workOrder : workOrderList) {
            String userNames = workOrder.getNextUserName();
            if (StringUtils.isNotBlank(userNames)) {
                WorkOrderNotice notice = mongoTemplate.findById(workOrder.getId(), WorkOrderNotice.class);
                String[] userNameArgs = StringUtils.split(userNames, ",");
                List<String> noticeUsers = notice == null ? new ArrayList<>() : notice.getUsers();
                List<String> newNoticeUsers = new ArrayList<>();
                String content = String.format("工单：%s，工单id：%s，已经超时，请您尽快处理。", workOrder.getTitle(), workOrder.getId());
                boolean noticeFlag = false;
                for (String userName : userNameArgs) {
                    if (!noticeUsers.contains(userName)) {
                        noticeUsers.add(userName);
                        newNoticeUsers.add(userName);
                        noticeFlag = true;
                    }
                }
                if (noticeFlag) {
                    noticeService.send(content, null, newNoticeUsers);
                    notice = new WorkOrderNotice();
                    notice.setId(workOrder.getId());
                    notice.setUsers(noticeUsers);
                    notice.setContent(content);
                    mongoTemplate.save(notice);
                }

            }
        }
        log.info("超时通知同步结束");


    }



    private void validateAndFillWorkOrder(WorkOrderInfo workOrder) {
        if (StringUtils.isBlank(workOrder.getHandleUserName())) {
            throw new IllegalArgumentException("处理人为空");
        }
        if (StringUtils.isBlank(workOrder.getType())) {
            throw new IllegalArgumentException("工单类型为空");
        }
        if (StringUtils.isBlank(workOrder.getDataType())) {
            throw new IllegalArgumentException("资产类型为空");
        }
        if (StringUtils.isBlank(workOrder.getTitle())) {
            throw new IllegalArgumentException("标题为空");
        }
        if (CollectionUtils.isEmpty(workOrder.getData()) &&
                (!StringUtils.equals(OrderTypeEnum.DATA_ADD.getName(), workOrder.getType()))) {
            throw new IllegalArgumentException("关联资产为空");
        }
        if (StringUtils.isBlank(workOrder.getStatus())) {
            workOrder.setStatus(OrderStatusTypeEnum.STATUS_TO_BE_START.getName());
        }
        workOrder.setUpdateTime(DateUtil.getDateTime());
        workOrder.setCreateTime(DateUtil.getDateTime());
        workOrder.setCreateUserName(LogManager.getUserName());
        workOrder.setCreateUserDname(LogManager.getDUserName());
        workOrder.setCreateTimeMill(System.currentTimeMillis());
        if (StringUtils.isBlank(workOrder.getId())) {
            workOrder.setId(IdUtil.generateId());
        }
    }

    private void fillWorkOrderData(WorkOrderInfo workOrder, WorkOrderData workOrderData) {
        if (StringUtils.isBlank(workOrderData.getId())) {
            workOrderData.setId(IdUtil.generateId());
        }
        workOrderData.setWorkOrderId(workOrder.getId());
        if (StringUtils.isBlank(workOrderData.getType())) {
            workOrderData.setType(workOrder.getType());
        }
        if (StringUtils.isBlank(workOrderData.getDataType())) {
            workOrderData.setDataType(workOrder.getDataType());
        }
        if (StringUtils.isBlank(workOrderData.getHandleStatus())) {
            workOrderData.setHandleStatus("否");
        }
        if (StringUtils.isBlank(workOrderData.getConfirmStatus())) {
            workOrderData.setConfirmStatus("未确认");
        }
    }

    private WorkOrderInfoVO convertMessage(WorkOrderInfo workOrder) {
        WorkOrderInfoVO workOrderVo = BeanUtil.map(workOrder, WorkOrderInfoVO.class);
        Query query = Query.query(Criteria.where("workOrderId").is(workOrder.getId()));
        List<WorkOrderData> dataList = mongoTemplate.find(query, WorkOrderData.class);
        if (!CollectionUtils.isEmpty(dataList)) {
            List<WorkOrderDataVO> dataVolist = dataList.stream().map(e -> BeanUtil.map(e, WorkOrderDataVO.class)).collect(Collectors.toList());
            workOrderVo.setData(dataVolist);
        }
        return workOrderVo;
    }


    private List<User> getUserNameByGroup(String groupName) {
        List<UserGroup> groupList = authService.getUserGroups();
        UserGroup group = groupList.stream().filter(e -> StringUtils.equals(groupName, e.getName())).findFirst().orElse(null);
        if (group == null) {
            return new ArrayList<>();
        }
        List<User> userList = authService.getUserListByGroupId(group.getId());
        if (CollectionUtils.isEmpty(userList)) {
            return new ArrayList<>();
        }
        return userList;
    }

    private boolean contains(String nextUsers, String userName) {
        return Pattern.compile("(^|,)" + userName + "(,|$)").matcher(nextUsers).find();
    }


}
