package com.keymobile.governworkorder.core.service.impl;

import com.keymobile.governworkorder.core.remote.AuthService;
import com.keymobile.governworkorder.core.service.NoticeService;
import com.keymobile.governworkorder.core.util.ObjectUtil;
import com.keymobile.notice.common.NoticeInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.keymobile.notice.common.mq.NoticeMQConstant.NOTICE_EXCHANGE;
import static com.keymobile.notice.common.mq.NoticeMQConstant.NOTICE_ROUTING_KEY;

@Service
public class NoticeServiceImpl implements NoticeService {
    private static final Logger log = LoggerFactory.getLogger(NoticeServiceImpl.class);
    @Autowired
    private RabbitTemplate rabbitTemplate;

    @Autowired
    private AuthService authService;

    @Override
    public void send(String content, String url, List<String> userNames) {
        try {
            log.info("发送通知，内容 ：{}，用户：{}", content, userNames);
            List<String> userIds = getUserIds(userNames);
            if(CollectionUtils.isEmpty(userIds)){
                return;
            }
            NoticeInfo notice = convert(content, url, userIds);
            log.info("发送通知，exchange ：{}，ROUTINGKEY：{}，notice：{}", NOTICE_EXCHANGE, NOTICE_ROUTING_KEY, ObjectUtil.toJson(notice));
            rabbitTemplate.convertAndSend(NOTICE_EXCHANGE, NOTICE_ROUTING_KEY, notice);
        } catch (Exception e) {
            log.error("发送通知失败", e);
        }
    }

    private NoticeInfo convert(String content, String url, List<String> userIds) {
        return new NoticeInfo() {
            @Override
            public String getAppType() {
                return "index";
            }

            @Override
            public String getContent() {
                return content;
            }

            @Override
            public String getUrl() {
                return url;
            }

            @Override
            public Long getNoticeMillisecondTimestamp() {
                return System.currentTimeMillis();
            }

            @Override
            public List<String> getUserIdList() {
                return userIds;
            }
        };

    }

    private List<String> getUserIds(List<String> userNames) {
        if (CollectionUtils.isEmpty(userNames)) {
            return Collections.emptyList();
        }
        return userNames.stream().map(o -> {
            Map<String, Object> user = authService.exactFindByName(o);
            if (user == null) {
                return null;
            } else {
                return ObjectUtil.toString(user.get("id"));
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

}
