package com.keymobile.governworkorder.runtime.conf;

import com.mongodb.*;
import com.mongodb.client.gridfs.GridFSBucket;
import com.mongodb.client.gridfs.GridFSBuckets;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.config.AbstractMongoClientConfiguration;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.convert.*;
import org.springframework.data.mongodb.core.mapping.MongoMappingContext;
import org.springframework.data.mongodb.gridfs.GridFsTemplate;
import org.springframework.web.util.UriUtils;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

@Configuration
public class MongoConfig extends AbstractMongoClientConfiguration {
    @Value("${mongodb.uri}")
    private String host;
    @Value("${mongodb.database}")
    private String database;
    @Value("${mongodb.username}")
    private String user;
    @Value("${mongodb.password}")
    private String pswd;
    @Value("${mongodb.maxConnectionIdleTime}")
    private final Integer maxConnectionIdleTime = 60000;
    @Value("${mongodb.gridfs.bucket:g_work_order_file}")
    private String fileCollectionName;

    @Override
    protected String getDatabaseName() {
        return database;
    }

    @Bean
    public GridFSBucket gridFSBucket(MongoTemplate mongoTemplate) {
        return GridFSBuckets.create(mongoTemplate.getDb(), fileCollectionName);
    }

    @Bean
    public GridFsTemplate gridFsTemplate(MongoDatabaseFactory dbFactory, MongoConverter converter) {
        return new GridFsTemplate(dbFactory, converter, fileCollectionName);
    }

    @Override
    protected void configureClientSettings(MongoClientSettings.Builder builder) {

        if (host.split(",").length > 1) {
            String[] serverArray = host.split(",");
            List<ServerAddress> serverList = new ArrayList<>();
            for (String ipPort : serverArray) {
                String[] ipAndPort = ipPort.split(":");
                serverList.add(new ServerAddress(ipAndPort[0].trim(), Integer.parseInt(ipAndPort[1])));
            }
            builder.applyToClusterSettings(settings -> {
                settings.hosts(serverList);
            });
            builder.applyToConnectionPoolSettings(settings -> {
                settings.maxConnectionIdleTime(maxConnectionIdleTime, TimeUnit.MICROSECONDS);
            });
            builder.readPreference(ReadPreference.primaryPreferred());
            String password = UriUtils.decode(pswd, StandardCharsets.UTF_8);
            builder.credential(MongoCredential.createCredential(user, database, password.toCharArray()));

        } else {
            String uri = String.format("mongodb://%s:%s@%s/%s", user, pswd, host, database);
            builder.applyConnectionString(new ConnectionString(uri));
            builder.applyToConnectionPoolSettings(settings -> {
                settings.maxConnectionIdleTime(maxConnectionIdleTime, TimeUnit.MICROSECONDS);
            });
        }
    }


    @Bean
    public MappingMongoConverter mappingMongoConverter(MongoDatabaseFactory databaseFactory, MongoCustomConversions customConversions, MongoMappingContext mappingContext) {
        DbRefResolver dbRefResolver = new DefaultDbRefResolver(databaseFactory);
        MappingMongoConverter converter = new MappingMongoConverter(dbRefResolver, mappingContext);
        converter.setCustomConversions(customConversions);
        converter.setCodecRegistryProvider(databaseFactory);
        converter.setTypeMapper(new DefaultMongoTypeMapper(null));
        return converter;
    }

}

