package com.keymobile.governworkorder.core.util;

import org.springframework.beans.BeanUtils;

import java.util.List;
import java.util.stream.Collectors;

public class BeanUtil {

    private BeanUtil() {
        // 避免实例化
    }

    /**
     * 将源对象的属性映射到目标类型的新实例。
     *
     * @param source          源对象
     * @param destinationType 目标类型的Class对象
     * @param <S>             源对象的类型
     * @param <D>             目标对象的类型
     * @return 目标类型的新实例
     */
    public static <S, D> D map(S source, Class<D> destinationType) {
        D destination;
        try {
            destination = destinationType.newInstance();
            BeanUtils.copyProperties(source, destination);
            return destination;
        } catch (InstantiationException | IllegalAccessException e) {
            throw new RuntimeException("Failed to copy properties", e);
        }
    }

    /**
     * 将源对象的属性映射到目标对象。
     *
     * @param source       源对象
     * @param destination  目标对象
     * @param <S>          源对象的类型
     * @param <D>          目标对象的类型
     * @return 目标对象
     */
    public static <S, D> void map(S source, D destination) {
        if (source == null || destination == null) {
            throw new IllegalArgumentException("Source or destination cannot be null");
        }
        BeanUtils.copyProperties(source, destination);
    }


    /**
     * 将源对象列表的属性映射到目标类型对象的列表。
     *
     * @param sourceList      源对象列表
     * @param destinationType 目标类型的Class对象
     * @param <S>             源对象的类型
     * @param <D>             目标对象的类型
     * @return 目标类型对象的列表
     */
    public static <S, D> List<D> mapList(List<S> sourceList, Class<D> destinationType) {
        return sourceList.stream()
                .map(source -> map(source, destinationType))
                .collect(Collectors.toList());
    }
}
