package com.keymobile.governworkorder.core.service.impl;

import com.keymobile.governworkorder.core.persistence.model.FileData;
import com.keymobile.governworkorder.core.service.FileService;
import com.mongodb.client.gridfs.GridFSBucket;
import com.mongodb.client.gridfs.model.GridFSFile;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.bson.types.ObjectId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.gridfs.GridFsTemplate;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class FileServiceImpl implements FileService {

    private static final Logger logger = LoggerFactory.getLogger(FileService.class);

    @Autowired
    private GridFsTemplate gridFsTemplate;

    @Autowired
    private GridFSBucket gridFSBucket;
    @Autowired
    private MongoTemplate mongoTemplate;

    public static final String FILE_CONTENT_TYPE = "fileContentType";


    public GridFSFile retrieveFile(String fileId) {
        return gridFsTemplate.findOne(new Query(Criteria.where("_id").is(fileId)));
    }

    public FileData upload(MultipartFile file){

        try {
            Map<String, String> metadata = new HashMap<>();
            metadata.put(FILE_CONTENT_TYPE , file.getContentType());
            ObjectId fileId = gridFsTemplate.store(file.getInputStream(), file.getOriginalFilename(), file.getContentType(), metadata);
            return  new FileData(fileId.toString(), file.getOriginalFilename());
        } catch (IOException e) {
            logger.error("上传文件异常 - ", e);
            throw new RuntimeException(e);
        }

    }

    public void del(String id) {
        Query query = new Query(Criteria.where("_id").is(new ObjectId(id)));
        gridFsTemplate.delete(query);
    }

    public void download(String id, HttpServletResponse response) {
        try {
            // 查询文件
            GridFSFile gridFSFile = retrieveFile(id);
            if (gridFSFile == null) {
                throw new RuntimeException("文件不存在");
            }
            // 设置响应头
            String fileName = gridFSFile.getFilename();
            String contentType = gridFSFile.getMetadata().getString("contentType");
            String encodedFileName = URLEncoder.encode(fileName, "UTF-8");
            response.setContentType(contentType);
            response.setHeader("Content-Disposition", "attachment; filename=\"" + encodedFileName + "\"");
            response.setCharacterEncoding("utf-8");

            // 获取文件内容并写入响应的输出流
            try (InputStream inputStream = gridFSBucket.openDownloadStream(new ObjectId(id));
                 BufferedInputStream bis = new BufferedInputStream(inputStream);
                 ServletOutputStream outputStream = response.getOutputStream()) {

                byte[] buffer = new byte[2048];
                int len;
                while ((len = bis.read(buffer)) > 0) {
                    outputStream.write(buffer, 0, len);
                }
            }

        } catch (Exception e) {
            logger.error("FileService-download - ", e);
            throw new RuntimeException("文件下载失败");
        }
    }

    public List<Document> find(String fileId){
        Query query = new Query();
        if(StringUtils.isNotBlank(fileId)){
            query.addCriteria(Criteria.where("_id").is(fileId));
        }
        List<Document> list = mongoTemplate.find(query,Document.class,"g_work_order_file.files");
        list.stream().forEach(e->e.put("_id",e.get("_id").toString()));
        return list;
    }

    public byte[] getBytes(String id) {
        try {
            // 查询文件
            GridFSFile gridFSFile = retrieveFile(id);
            if (gridFSFile == null) {
                throw new RuntimeException("文件不存在");
            }

 /*           // 设置文件名
            String fileName = gridFSFile.getFilename();
            // 文件名编码用于响应头（如果需要）
            String encodedFileName = URLEncoder.encode(fileName, "UTF-8");*/

            // 读取文件内容到字节数组
            try (InputStream inputStream = gridFSBucket.openDownloadStream(new ObjectId(id));
                 BufferedInputStream bis = new BufferedInputStream(inputStream);
                 ByteArrayOutputStream bos = new ByteArrayOutputStream()) {

                byte[] buffer = new byte[2048];
                int len;
                while ((len = bis.read(buffer)) > 0) {
                    bos.write(buffer, 0, len);
                }

                return bos.toByteArray();
            }

        } catch (Exception e) {
            logger.error("FileService-download - ", e);
            throw new RuntimeException("文件下载失败");
        }
    }


}
